"use client"

import { useState } from "react"
import { useForm } from "react-hook-form"
import { zodResolver } from "@hookform/resolvers/zod"
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
  DialogFooter,
} from "@/components/ui/dialog"
import { Button } from "@/components/ui/button"
import { Textarea } from "@/components/ui/textarea"
import { Label } from "@/components/ui/label"
import { AlertTriangle } from "lucide-react"
import { blockAgentSchema, type BlockAgentInput } from "@/lib/validations/agent"
import type { Agent } from "@/lib/data/agents"

interface BlockAgentDialogProps {
  agent: Agent | null
  open: boolean
  onOpenChange: (open: boolean) => void
  onConfirm: (reason: string) => void
}

export function BlockAgentDialog({ agent, open, onOpenChange, onConfirm }: BlockAgentDialogProps) {
  const [isSubmitting, setIsSubmitting] = useState(false)

  const {
    register,
    handleSubmit,
    reset,
    formState: { errors },
  } = useForm<BlockAgentInput>({
    resolver: zodResolver(blockAgentSchema),
    defaultValues: { reason: "" },
  })

  const onSubmit = async (data: BlockAgentInput) => {
    setIsSubmitting(true)
    await new Promise((resolve) => setTimeout(resolve, 300))
    onConfirm(data.reason)
    reset()
    setIsSubmitting(false)
  }

  const handleOpenChange = (newOpen: boolean) => {
    if (!newOpen) reset()
    onOpenChange(newOpen)
  }

  if (!agent) return null

  return (
    <Dialog open={open} onOpenChange={handleOpenChange}>
      <DialogContent className="sm:max-w-md">
        <DialogHeader>
          <div className="flex items-center gap-2">
            <div className="flex h-10 w-10 items-center justify-center rounded-full bg-destructive/10">
              <AlertTriangle className="h-5 w-5 text-destructive" aria-hidden="true" />
            </div>
            <div>
              <DialogTitle>Block Agent</DialogTitle>
              <DialogDescription>
                Block {agent.firstName} {agent.lastName}
              </DialogDescription>
            </div>
          </div>
        </DialogHeader>

        <form onSubmit={handleSubmit(onSubmit)} className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="block-reason">
              Reason for blocking <span className="text-destructive">*</span>
            </Label>
            <Textarea
              id="block-reason"
              placeholder="Please provide a detailed reason for blocking this agent..."
              className="min-h-[120px] resize-none"
              {...register("reason")}
              aria-invalid={errors.reason ? "true" : "false"}
              aria-describedby={errors.reason ? "block-reason-error" : undefined}
            />
            {errors.reason && (
              <p id="block-reason-error" className="text-sm text-destructive" role="alert">
                {errors.reason.message}
              </p>
            )}
          </div>

          <DialogFooter className="gap-2 sm:gap-0">
            <Button type="button" variant="outline" onClick={() => handleOpenChange(false)} disabled={isSubmitting}>
              Cancel
            </Button>
            <Button type="submit" variant="destructive" disabled={isSubmitting}>
              {isSubmitting ? "Blocking..." : "Block Agent"}
            </Button>
          </DialogFooter>
        </form>
      </DialogContent>
    </Dialog>
  )
}
