"use client"

import { useState, useEffect } from "react"
import { useRouter } from "next/navigation"
import { useForm } from "react-hook-form"
import { zodResolver } from "@hookform/resolvers/zod"
import { ArrowLeft, Plus, Trash2, Edit2 } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Textarea } from "@/components/ui/textarea"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog"
import { fixPackageSchema, type FixPackageInput } from "@/lib/validations/fix-packages"
import { fixPackageDataStore, type PackageVehicle } from "@/lib/data/fix-packages"
import { useToastEnhanced } from "@/hooks/use-toast-enhanced"
import DOMPurify from "dompurify"

interface PackageFormClientProps {
  mode: "add" | "edit"
  packageId?: string
}

export function PackageFormClient({ mode, packageId }: PackageFormClientProps) {
  const router = useRouter()
  const toast = useToastEnhanced()
  const [isSubmitting, setIsSubmitting] = useState(false)
  const [vehicles, setVehicles] = useState<PackageVehicle[]>([])
  const [isVehicleDialogOpen, setIsVehicleDialogOpen] = useState(false)
  const [editingVehicle, setEditingVehicle] = useState<PackageVehicle | null>(null)
  const [vehicleIdCounter, setVehicleIdCounter] = useState(1)

  const {
    register,
    handleSubmit,
    setValue,
    formState: { errors },
  } = useForm<FixPackageInput>({
    resolver: zodResolver(fixPackageSchema),
    defaultValues: {
      name: "",
      locationDetails: "",
      itineraryDetails: "",
      otherDetails: "",
      originalPrice: 0,
      discountedPrice: 0,
      totalDays: 1,
      vehicles: [],
    },
  })

  // Load existing package data in edit mode
  useEffect(() => {
    if (mode === "edit" && packageId) {
      const pkg = fixPackageDataStore.getPackageById(packageId)
      if (pkg) {
        setValue("name", pkg.name)
        setValue("locationDetails", pkg.locationDetails)
        setValue("itineraryDetails", pkg.itineraryDetails)
        setValue("otherDetails", pkg.otherDetails)
        setValue("originalPrice", pkg.originalPrice)
        setValue("discountedPrice", pkg.discountedPrice)
        setValue("totalDays", pkg.totalDays)
        setVehicles(pkg.vehicles)
        setValue("vehicles", pkg.vehicles)
        if (pkg.vehicles.length > 0) {
          const maxId = Math.max(...pkg.vehicles.map((v) => Number.parseInt(v.id.replace("v", "")) || 0))
          setVehicleIdCounter(maxId + 1)
        }
      }
    }
  }, [mode, packageId, setValue])

  // Update form value when vehicles change
  useEffect(() => {
    setValue("vehicles", vehicles)
  }, [vehicles, setValue])

  const onSubmit = async (data: FixPackageInput) => {
    setIsSubmitting(true)
    try {
      const sanitizedData: FixPackageInput = {
        name: DOMPurify.sanitize(data.name.trim()),
        locationDetails: DOMPurify.sanitize(data.locationDetails.trim()),
        itineraryDetails: DOMPurify.sanitize(data.itineraryDetails.trim()),
        otherDetails: DOMPurify.sanitize(data.otherDetails.trim()),
        originalPrice: data.originalPrice,
        discountedPrice: data.discountedPrice,
        totalDays: data.totalDays,
        vehicles: vehicles,
      }

      if (mode === "add") {
        const newPackage = fixPackageDataStore.addPackage(
          sanitizedData.name,
          sanitizedData.locationDetails,
          sanitizedData.itineraryDetails,
          sanitizedData.otherDetails,
          sanitizedData.originalPrice,
          sanitizedData.discountedPrice,
          sanitizedData.totalDays,
          sanitizedData.vehicles,
        )
        if (newPackage) {
          toast.success({
            title: "Package Added",
            description: `${sanitizedData.name} has been added successfully.`,
          })
          router.push("/admin/fix-packages")
        } else {
          toast.error({
            title: "Failed to Add Package",
            description: "A package with this name already exists.",
          })
        }
      } else if (mode === "edit" && packageId) {
        const updated = fixPackageDataStore.updatePackage(
          packageId,
          sanitizedData.name,
          sanitizedData.locationDetails,
          sanitizedData.itineraryDetails,
          sanitizedData.otherDetails,
          sanitizedData.originalPrice,
          sanitizedData.discountedPrice,
          sanitizedData.totalDays,
          sanitizedData.vehicles,
        )
        if (updated) {
          toast.success({
            title: "Package Updated",
            description: `${sanitizedData.name} has been updated successfully.`,
          })
          router.push("/admin/fix-packages")
        } else {
          toast.error({
            title: "Failed to Update Package",
            description: "A package with this name already exists.",
          })
        }
      }
    } finally {
      setIsSubmitting(false)
    }
  }

  const handleAddVehicle = (vehicle: Omit<PackageVehicle, "id">) => {
    const newVehicle: PackageVehicle = {
      ...vehicle,
      id: `v${vehicleIdCounter}`,
    }
    setVehicles([...vehicles, newVehicle])
    setVehicleIdCounter(vehicleIdCounter + 1)
    setIsVehicleDialogOpen(false)
    toast.success({
      title: "Vehicle Added",
      description: `${vehicle.vehicleName} has been added to the package.`,
    })
  }

  const handleEditVehicle = (vehicle: Omit<PackageVehicle, "id">) => {
    if (!editingVehicle) return
    const updatedVehicles = vehicles.map((v) => (v.id === editingVehicle.id ? { ...vehicle, id: v.id } : v))
    setVehicles(updatedVehicles)
    setEditingVehicle(null)
    setIsVehicleDialogOpen(false)
    toast.success({
      title: "Vehicle Updated",
      description: `${vehicle.vehicleName} has been updated.`,
    })
  }

  const handleDeleteVehicle = (id: string) => {
    const vehicle = vehicles.find((v) => v.id === id)
    setVehicles(vehicles.filter((v) => v.id !== id))
    toast.success({
      title: "Vehicle Removed",
      description: `${vehicle?.vehicleName} has been removed from the package.`,
    })
  }

  const openEditDialog = (vehicle: PackageVehicle) => {
    setEditingVehicle(vehicle)
    setIsVehicleDialogOpen(true)
  }

  return (
    <section className="admin-section">
      <header className="admin-page-header mb-6">
        <div className="flex items-center gap-4">
          <Button variant="ghost" size="icon" onClick={() => router.back()} aria-label="Go back">
            <ArrowLeft className="h-5 w-5" />
          </Button>
          <div>
            <h1 className="admin-page-title">{mode === "add" ? "Add New Package" : "Edit Package"}</h1>
            <p className="admin-page-description">
              {mode === "add"
                ? "Create a new fix package with pricing, itinerary, and vehicle details"
                : "Update package information and manage vehicles"}
            </p>
          </div>
        </div>
      </header>

      <form onSubmit={handleSubmit(onSubmit)} className="space-y-6">
        <Card>
          <CardHeader>
            <CardTitle>Package Information</CardTitle>
            <CardDescription>Enter the basic details of the package</CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid gap-2">
              <Label htmlFor="package-name">Package Name</Label>
              <Input
                id="package-name"
                placeholder="e.g., Char Dham Yatra"
                {...register("name")}
                aria-invalid={!!errors.name}
                aria-describedby={errors.name ? "package-name-error" : undefined}
              />
              {errors.name && (
                <p id="package-name-error" className="text-sm text-destructive">
                  {errors.name.message}
                </p>
              )}
            </div>

            <div className="grid gap-2">
              <Label htmlFor="location-details">Location Details</Label>
              <Textarea
                id="location-details"
                placeholder="e.g., Yamunotri, Gangotri, Kedarnath, Badrinath - Uttarakhand"
                rows={2}
                {...register("locationDetails")}
                aria-invalid={!!errors.locationDetails}
                aria-describedby={errors.locationDetails ? "location-details-error" : undefined}
              />
              {errors.locationDetails && (
                <p id="location-details-error" className="text-sm text-destructive">
                  {errors.locationDetails.message}
                </p>
              )}
            </div>

            <div className="grid gap-2">
              <Label htmlFor="itinerary-details">Itinerary Details</Label>
              <Textarea
                id="itinerary-details"
                placeholder="Day-wise itinerary with activities and destinations..."
                rows={4}
                {...register("itineraryDetails")}
                aria-invalid={!!errors.itineraryDetails}
                aria-describedby={errors.itineraryDetails ? "itinerary-details-error" : undefined}
              />
              {errors.itineraryDetails && (
                <p id="itinerary-details-error" className="text-sm text-destructive">
                  {errors.itineraryDetails.message}
                </p>
              )}
            </div>

            <div className="grid gap-2">
              <Label htmlFor="other-details">Other Details</Label>
              <Textarea
                id="other-details"
                placeholder="Package inclusions, exclusions, terms and conditions..."
                rows={4}
                {...register("otherDetails")}
                aria-invalid={!!errors.otherDetails}
                aria-describedby={errors.otherDetails ? "other-details-error" : undefined}
              />
              {errors.otherDetails && (
                <p id="other-details-error" className="text-sm text-destructive">
                  {errors.otherDetails.message}
                </p>
              )}
            </div>

            <div className="grid grid-cols-1 sm:grid-cols-3 gap-4">
              <div className="grid gap-2">
                <Label htmlFor="original-price">Original Price (₹)</Label>
                <Input
                  id="original-price"
                  type="number"
                  placeholder="50000"
                  {...register("originalPrice", { valueAsNumber: true })}
                  aria-invalid={!!errors.originalPrice}
                  aria-describedby={errors.originalPrice ? "original-price-error" : undefined}
                />
                {errors.originalPrice && (
                  <p id="original-price-error" className="text-sm text-destructive">
                    {errors.originalPrice.message}
                  </p>
                )}
              </div>

              <div className="grid gap-2">
                <Label htmlFor="discounted-price">Discounted Price (₹)</Label>
                <Input
                  id="discounted-price"
                  type="number"
                  placeholder="42000"
                  {...register("discountedPrice", { valueAsNumber: true })}
                  aria-invalid={!!errors.discountedPrice}
                  aria-describedby={errors.discountedPrice ? "discounted-price-error" : undefined}
                />
                {errors.discountedPrice && (
                  <p id="discounted-price-error" className="text-sm text-destructive">
                    {errors.discountedPrice.message}
                  </p>
                )}
              </div>

              <div className="grid gap-2">
                <Label htmlFor="total-days">Total Days</Label>
                <Input
                  id="total-days"
                  type="number"
                  placeholder="7"
                  {...register("totalDays", { valueAsNumber: true })}
                  aria-invalid={!!errors.totalDays}
                  aria-describedby={errors.totalDays ? "total-days-error" : undefined}
                />
                {errors.totalDays && (
                  <p id="total-days-error" className="text-sm text-destructive">
                    {errors.totalDays.message}
                  </p>
                )}
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <div className="flex items-center justify-between">
              <div>
                <CardTitle>Package Vehicles</CardTitle>
                <CardDescription>Manage vehicles available for this package</CardDescription>
              </div>
              <Button
                type="button"
                variant="outline"
                onClick={() => {
                  setEditingVehicle(null)
                  setIsVehicleDialogOpen(true)
                }}
              >
                <Plus className="mr-2 h-4 w-4" />
                Add Vehicle
              </Button>
            </div>
          </CardHeader>
          <CardContent>
            {vehicles.length === 0 ? (
              <div className="text-center py-8 text-muted-foreground">
                <p>No vehicles added yet. Click &quot;Add Vehicle&quot; to get started.</p>
              </div>
            ) : (
              <div className="grid gap-4 sm:grid-cols-2 lg:grid-cols-3">
                {vehicles.map((vehicle) => (
                  <Card key={vehicle.id} className="overflow-hidden">
                    <div className="aspect-video relative bg-muted">
                      <img
                        src={vehicle.photo || "/placeholder.svg"}
                        alt={vehicle.vehicleName}
                        className="object-cover w-full h-full"
                      />
                    </div>
                    <CardContent className="p-4 space-y-2">
                      <h3 className="font-semibold">{vehicle.vehicleName}</h3>
                      <div className="flex items-center justify-between text-sm">
                        <span className="text-muted-foreground">Per Day:</span>
                        <span className="font-medium">₹{vehicle.perDayPrice.toLocaleString()}</span>
                      </div>
                      <div className="flex items-center justify-between text-sm">
                        <span className="text-muted-foreground">Passengers:</span>
                        <span className="font-medium">{vehicle.passengerLimit}</span>
                      </div>
                      <div className="flex gap-2 pt-2">
                        <Button
                          type="button"
                          variant="outline"
                          size="sm"
                          onClick={() => openEditDialog(vehicle)}
                          className="flex-1"
                        >
                          <Edit2 className="mr-1 h-3 w-3" />
                          Edit
                        </Button>
                        <Button
                          type="button"
                          variant="outline"
                          size="sm"
                          onClick={() => handleDeleteVehicle(vehicle.id)}
                          className="flex-1 text-destructive hover:text-destructive"
                        >
                          <Trash2 className="mr-1 h-3 w-3" />
                          Delete
                        </Button>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            )}
          </CardContent>
        </Card>

        <div className="flex justify-end gap-4">
          <Button type="button" variant="outline" onClick={() => router.back()}>
            Cancel
          </Button>
          <Button type="submit" disabled={isSubmitting}>
            {isSubmitting
              ? mode === "add"
                ? "Adding..."
                : "Updating..."
              : mode === "add"
                ? "Add Package"
                : "Update Package"}
          </Button>
        </div>
      </form>

      <VehicleDialog
        open={isVehicleDialogOpen}
        onOpenChange={(open) => {
          setIsVehicleDialogOpen(open)
          if (!open) setEditingVehicle(null)
        }}
        onSave={editingVehicle ? handleEditVehicle : handleAddVehicle}
        vehicle={editingVehicle}
      />
    </section>
  )
}

interface VehicleDialogProps {
  open: boolean
  onOpenChange: (open: boolean) => void
  onSave: (vehicle: Omit<PackageVehicle, "id">) => void
  vehicle: PackageVehicle | null
}

function VehicleDialog({ open, onOpenChange, onSave, vehicle }: VehicleDialogProps) {
  const [photo, setPhoto] = useState(vehicle?.photo || "")
  const [vehicleName, setVehicleName] = useState(vehicle?.vehicleName || "")
  const [perDayPrice, setPerDayPrice] = useState(vehicle?.perDayPrice || 0)
  const [passengerLimit, setPassengerLimit] = useState(vehicle?.passengerLimit || 1)
  const [errors, setErrors] = useState<Record<string, string>>({})

  useEffect(() => {
    if (vehicle) {
      setPhoto(vehicle.photo)
      setVehicleName(vehicle.vehicleName)
      setPerDayPrice(vehicle.perDayPrice)
      setPassengerLimit(vehicle.passengerLimit)
    } else {
      setPhoto("")
      setVehicleName("")
      setPerDayPrice(0)
      setPassengerLimit(1)
    }
    setErrors({})
  }, [vehicle, open])

  const handleSave = () => {
    const newErrors: Record<string, string> = {}

    if (!photo.trim()) {
      newErrors.photo = "Photo is required"
    }
    if (!vehicleName.trim() || vehicleName.length < 2) {
      newErrors.vehicleName = "Vehicle name must be at least 2 characters"
    }
    if (perDayPrice < 100) {
      newErrors.perDayPrice = "Per day price must be at least ₹100"
    }
    if (passengerLimit < 1) {
      newErrors.passengerLimit = "Passenger limit must be at least 1"
    }

    if (Object.keys(newErrors).length > 0) {
      setErrors(newErrors)
      return
    }

    onSave({
      photo: DOMPurify.sanitize(photo.trim()),
      vehicleName: DOMPurify.sanitize(vehicleName.trim()),
      perDayPrice,
      passengerLimit,
    })
  }

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-md">
        <DialogHeader>
          <DialogTitle>{vehicle ? "Edit Vehicle" : "Add Vehicle"}</DialogTitle>
          <DialogDescription>
            {vehicle ? "Update the vehicle details" : "Add a new vehicle to this package"}
          </DialogDescription>
        </DialogHeader>
        <div className="space-y-4">
          <div className="grid gap-2">
            <Label htmlFor="vehicle-photo">Vehicle Photo URL</Label>
            <Input
              id="vehicle-photo"
              placeholder="/images/vehicle.jpg or URL"
              value={photo}
              onChange={(e) => setPhoto(e.target.value)}
              aria-invalid={!!errors.photo}
            />
            {errors.photo && <p className="text-sm text-destructive">{errors.photo}</p>}
          </div>

          <div className="grid gap-2">
            <Label htmlFor="vehicle-name">Vehicle Name</Label>
            <Input
              id="vehicle-name"
              placeholder="e.g., Luxury SUV"
              value={vehicleName}
              onChange={(e) => setVehicleName(e.target.value)}
              aria-invalid={!!errors.vehicleName}
            />
            {errors.vehicleName && <p className="text-sm text-destructive">{errors.vehicleName}</p>}
          </div>

          <div className="grid gap-2">
            <Label htmlFor="per-day-price">Per Day Price (₹)</Label>
            <Input
              id="per-day-price"
              type="number"
              placeholder="5000"
              value={perDayPrice}
              onChange={(e) => setPerDayPrice(Number(e.target.value))}
              aria-invalid={!!errors.perDayPrice}
            />
            {errors.perDayPrice && <p className="text-sm text-destructive">{errors.perDayPrice}</p>}
          </div>

          <div className="grid gap-2">
            <Label htmlFor="passenger-limit">Passenger Limit</Label>
            <Input
              id="passenger-limit"
              type="number"
              placeholder="7"
              value={passengerLimit}
              onChange={(e) => setPassengerLimit(Number(e.target.value))}
              aria-invalid={!!errors.passengerLimit}
            />
            {errors.passengerLimit && <p className="text-sm text-destructive">{errors.passengerLimit}</p>}
          </div>
        </div>
        <DialogFooter>
          <Button type="button" variant="outline" onClick={() => onOpenChange(false)}>
            Cancel
          </Button>
          <Button type="button" onClick={handleSave}>
            {vehicle ? "Update Vehicle" : "Add Vehicle"}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  )
}
